/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef UUID_a7efdd62_01c8_11e4_9403_f36b20d3dfa8
#define UUID_a7efdd62_01c8_11e4_9403_f36b20d3dfa8

/**
 * \file vncwifidisplaysink.h
 *
 * This file defines SDK API methods that VNC WiFi Display Sink instances may
 * use. You should normally include vncwifidisplaysdk.h instead of including
 * this file directly.
 *
 * This SDK does not currently support secondary sinks, or coupled sink
 * operation. A VNC WiFi Display Sink instance may only act as a primary sink.
 * Future versions of this SDK may include support for secondary sinks and
 * coupled sink operation may be added in a future.
 */

#include "vncwifidisplaydecoder.h"
#include "vncwifidisplayparameters.h"
#include "vncwifidisplaysinkcallbacks.h"
#include "vncwifidisplaytypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Create a new VNC WiFi Display Sink instance.
 *
 * This method creates and returns a new, uninitialized VNC WiFi Display Sink
 * instance. All of the sink's parameters have their default values, and the
 * WiFi Display sink thread will not be started until you call
 * VNCWiFiDisplaySinkStart(). Before you can start this sink, you must first
 * perform the following steps:
 *
 * - Call VNCWiFiDisplaySinkAddLicense() to add at least one valid license.
 * - Call VNCWiFiDisplaySinkSetNativeResolution() to set your native
 * resolution.
 * - Call VNCWiFiDisplaySinkSetActiveConnectorType() to set your active display
 * connector type.
 * - Call VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile() to register a decoder plugin.
 * - If audio support is required, call VNCWiFiDisplaySinkAddAudioCodec() to
 * add at least one audio codec, supported by the decoder plugin, to the sink.
 * - Call VNCWiFiDisplaySinkAddH264Codec() to add at least one H.264 codec,
 * supported by the decoder plugin, to the sink.
 * - Call VNCWiFiDisplaySinkSetSource() to specify the WiFi Display source to
 * connect to.
 *
 * If you are implementing a MirrorLink client, then you must also set an
 * appropriate configuration for the User Input Back Channel (UIBC) MirrorLink
 * connection with VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration() before
 * starting the sink.
 *
 * You are free to call VNCWiFiDisplaySinkStart() and VNCWiFiDisplaySinkStop()
 * multiple times on the same VNC WiFi Display Sink instance. Whilst the sink
 * is not running, you may adjust the codecs that you support, and change the
 * WiFi Display source that you wish to connect to.
 *
 * The SDK does not impose an upper limit on the number of VNC WiFi Display
 * Sink instances that may exist at the same time. However, each sink must
 * connect to a different WiFi Display source.
 *
 * At present, a VNC WiFi Display Sink may only implement a primary sink. The
 * sinkType parameter is reserved for future use, and must always be set to
 * VNCWiFiDisplaySinkTypePrimary in this version of the SDK.
 *
 * \param callbacks Pointer to a VNCWiFiDisplaySinkCallbacks structure holding
 * pointers to callbacks that may be invoked by the SDK. The SDK copies the
 * contents of *callbacks, so it is safe to free this structure once this
 * method returns. This parameter may not be NULL.
 * \param callbacksSize The size of *callbacks. You should pass
 * sizeof(VNCWiFiDisplaySinkCallbacks) here.
 * \param sinkType The type of this WiFi Display sink. This must be set to
 * VNCWiFiDisplaySinkTypePrimary.
 *
 * \return The new VNC WiFi Display Sink. When you are finished with it, call
 * VNCWiFiDisplaySinkDestroy() to destroy it. If a VNC WiFi Display Sink could
 * not be created, this method returns NULL.
 *
 * \see VNCWiFiDisplaySinkDestroy, VNCWiFiDisplaySinkCallbacks
 * \see VNCWiFiDisplaySinkSetContext, VNCWiFiDisplaySinkGetContext
 * \see VNCWiFiDisplaySinkAddLicense
 * \see VNCWiFiDisplaySinkSetNativeResolution
 * \see VNCWiFiDisplaySinkSetActiveConnectorType
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplaySinkAddH264Codec, VNCWiFiDisplaySinkSetSource
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkStart
 */
typedef VNCWiFiDisplaySink *VNCCALL
VNCWiFiDisplaySinkCreate(VNCWiFiDisplaySinkCallbacks *callbacks,
                         size_t callbacksSize,
                         VNCWiFiDisplaySinkType sinkType);

/**
 * \brief Destroy an instance of the VNC WiFi Display Sink.
 *
 * This method destroys a VNC WiFi Display Sink instance. If the VNC WiFi
 * Display Sink instance is still running, it will be stopped first.
 *
 * After calling this method, you should not attempt to interact further with
 * the VNC WiFi Display Sink.
 *
 * Note that you must never call this method directly from an SDK callback,
 * since this will result in a deadlock.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \see VNCWiFiDisplaySinkCreate, VNCWiFiDisplaySinkStop
 */
typedef void VNCCALL
VNCWiFiDisplaySinkDestroy(VNCWiFiDisplaySink *sink);

/**
 * \brief Associates a context pointer with a VNC WiFi Display Sink.
 *
 * Use this method to associate data with a VNC WiFi Display Sink instance so
 * that you can recover it at a later time. The context pointer is supplied to
 * all callbacks made by the SDK for this sink, and can also be retrieved by
 * calling VNCWiFiDisplaySinkGetContext().
 *
 * A second call to VNCWiFiDisplaySinkSetContext() replaces the context pointer
 * assigned by the first call.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer to associate with the device.
 *
 * \note The SDK will not interact in any way with the memory to which the
 * context pointer points.  In particular, the SDK will not free the memory
 * when the VNC WiFi Display Sink is destroyed.
 *
 * \see VNCWiFiDisplaySinkGetContext
 */
typedef void VNCCALL
VNCWiFiDisplaySinkSetContext(VNCWiFiDisplaySink *sink,
                             void *context);

/**
 * \brief Retrieves the VNC WiFi Display Sink's context pointer.
 *
 * This method returns the context pointer associated with the VNC WiFi Display
 * Sink by a call to VNCWiFiDisplaySinkSetContext().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \return The associated context pointer, or NULL if there is none.
 *
 * \see VNCWiFiDisplaySinkSetContext
 */
typedef void *VNCCALL
VNCWiFiDisplaySinkGetContext(VNCWiFiDisplaySink *sink);

/**
 * \brief Adds a VNC license to a VNC WiFi Display Sink.
 *
 * The VNC WiFi Display Sink must be licensed before it can be started. It is
 * recommended that this method is called soon after creating the VNC WiFi
 * Display Sink instance.
 *
 * To obtain a license for use with the VNC WiFi Display SDK, contact your
 * RealVNC sales representative.
 *
 * You must provide the text of the entire license to this method, including
 * the header and footer. If your license has been supplied to you in a
 * .vnclicense file, then you should provide the entire contents of that file.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param licenseText The text of the VNC license, as a null-terminated UTF-8
 * string.
 * \param serialNumber If not NULL, then, on successful return, set to contain
 * the license serial number, which is a big-endian UUID. If the serial number
 * is not required, then set this parameter to NULL.
 * \param serialNumberSize The size in bytes of the buffer pointed to by
 * serialNumber. This should be 16 if serialNumber is not NULL, and 0 if
 * serialNumber is NULL.
 *
 * \return VNCWiFiDisplayErrorNone The license was added successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The license could not be
 * added, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The license could not be added,
 * as licenseText is NULL or the combination of serialNumber and
 * serialNumberSize are invalid.
 * \return VNCWiFiDisplayErrorLicenseNotValid The license could not be added,
 * as licenseText does not contain a valid VNC license.
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkAddLicense(VNCWiFiDisplaySink *sink,
                             const char *licenseText,
                             vnc_uint8_t *serialNumber,
                             size_t serialNumberSize);

/**
 * \brief Queries a VNC WiFi Display Sink for the value of an integer-valued
 * runtime property.
 *
 * You can use this method to find out information about a running VNC WiFi
 * Display Sink, such as whether or not the WiFi Display sink thread is
 * running.
 *
 * To query string-valued runtime parameters, use
 * VNCWiFiDisplaySinkGetPropertyString() instead.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param property The property to query.
 *
 * \return The value of the property.
 *
 * \see VNCWiFiDisplayProperty
 */
typedef vnc_int32_t VNCCALL
VNCWiFiDisplaySinkGetProperty(VNCWiFiDisplaySink *sink,
                              VNCWiFiDisplayProperty property);

/**
 * \brief Queries a VNC WiFi Display Sink for the value of a string-valued
 * runtime property.
 *
 * You can use this method to find out information about a running VNC WiFi
 * Display Sink, such as whether or not the WiFi Display sink thread is
 * running.
 *
 * To query integer-valued runtime parameters, use
 * VNCWiFiDisplaySinkGetProperty() instead.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param property The property to query.
 *
 * \return The value of the property as a null-terminated UTF-8 string, or NULL
 * if the property value could not be determined. The caller is responsible for
 * freeing the string using VNCWiFiDisplayFreeString().
 *
 * \see VNCWiFiDisplayProperty
 */
typedef char *VNCCALL
VNCWiFiDisplaySinkGetPropertyString(VNCWiFiDisplaySink *sink,
                                    VNCWiFiDisplayProperty property);

/**
 * \brief Sets the value of a VNC WiFi Display Sink parameter.
 *
 * If the WiFi Display sink thread is running, then the new value will take
 * effect immediately, provided that this makes sense. Otherwise, the new value
 * will be used the next time the WiFi Display sink thread is started.
 *
 * VNC WiFi Display Sink parameters may never have NULL values. To 'clear' the
 * value of a string parameter, pass an empty string as the value.
 *
 * Parameters whose values are integers are formatted in decimal.
 *
 * VNC WiFi Display Sink parameters are case-insensitive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param parameter The name of the VNC WiFi Display parameter to set as a
 * null-terminated UTF-8 string. See vncwifidisplayparameters.h for an
 * explanation of the supported parameters.
 * \param value The new parameter value as a null-terminated UTF-8 string.
 *
 * \return VNCWiFiDisplayErrorNone The parameter was set successfully.
 * \return VNCWiFiDisplayErrorInvalidParameter Either the parameter name is
 * not recognized, or the given value is not valid for this parameter.
 *
 * \see VNCWiFiDisplaySinkGetParameter, vncwifidisplayparameters.h
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetParameter(VNCWiFiDisplaySink *sink,
                               const char *parameter,
                               const char *value);

/**
 * \brief Queries the value of a VNC WiFi Display Sink parameter.
 *
 * The configuration of a VNC WiFi Display Sink is stored internally as a list
 * of name-value pairs. This method queries the value of a VNC WiFi Display
 * Sink parameter and returns it. The parameter names are case-insensitive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param parameter The name of the VNC WiFi Display Sink parameter to query
 * as a null-terminated UTF-8 string. See vncwifidisplayparameters.h for an
 * explanation of the supported parameters.
 *
 * \return A copy of the parameter's value as a null-terminated UTF-8 string.
 * When you are finished with the copied value, you must free it by calling
 * VNCWiFiDisplayFreeString(). If the parameter name is invalid or memory
 * could not be allocated, then NULL is returned.
 *
 * \note The values of parameters whose values are integers are formatted into
 * strings of decimal digits.
 *
 * \see VNCWiFiDisplaySinkSetParameter, vncwifidisplayparameters.h
 */
typedef char *VNCCALL
VNCWiFiDisplaySinkGetParameter(VNCWiFiDisplaySink *sink,
                               const char *parameter);

/**
 * \brief Sets the native resolution for a VNC WiFi Display Sink.
 *
 * The resolution parameter must be set to a single value from the appropriate
 * resolution and refresh rates bitmap value group, depending on the chosen
 * VNCWiFiDisplayResolutionType.  For example, if resolutionType is
 * VNCWiFiDisplayResolutionTypeVESA, then resolution must be set to a single
 * value from the VESA bitmap value group (such as
 * VNCWiFiDisplayResolutionVESA_800_600_p60).
 *
 * You must call this method before invoking VNCWiFiDisplaySinkStart(). The
 * native resolution may not be changed whilst the VNC WiFi Display Sink is
 * running. 
 *
 * The native resolution is reported to the WiFi Display source through the
 * wfd-video-formats RTSP parameter.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param resolutionType The native resolution type 
 * \param resolution The native resolution, represented as a single value from
 * the appropriate resolution and refresh rates bitmap value group.
 *
 * \return VNCWiFiDisplayErrorNone The native resolution was set successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The native resolution could
 * not be set, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The native resolution could not
 * be set, as an invalid resolution type was specified, or an invalid
 * resolution value was specified for the resolution type.
 *
 * \see VNCWiFiDisplayResolutionType
 * \see ResolutionCEA
 * \see ResolutionVESA
 * \see ResolutionHH
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetNativeResolution(
                                   VNCWiFiDisplaySink *sink,
                                   VNCWiFiDisplayResolutionType resolutionType,
                                   vnc_uint32_t resolution);

/**
 * \brief Sets the active connector type for a VNC WiFi Display Sink.
 *
 * You should specify the type of connector being used to connect your
 * integrated or external display to your hardware platform.
 *
 * You must call this method before invoking VNCWiFiDisplaySinkStart(). You may
 * also call this method whilst the VNC WiFi Display Sink is running, if the
 * connector type changes (e.g. because a different display is now being used).
 *
 * The active connector type is reported to the WiFi Display source through the
 * wfd-connector-type RTSP parameter.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param connectorType The active connector type.
 *
 * \return VNCWiFiDisplayErrorNone The active connector type was set
 * successfully.
 * \return VNCWiFiDisplayErrorInvalidParameter The active connector type could
 * not be set, as an invalid connector type was specified.
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetActiveConnectorType(
                                    VNCWiFiDisplaySink *sink,
                                    VNCWiFiDisplayConnectorType connectorType);

/**
 * \brief Registers a VNC WiFi Display Decoder Plugin for a VNC WiFi Display
 * Sink.
 *
 * A VNC WiFi Display Decoder Plugin is responsible for decoding audio and/or
 * video streams received from the WiFi Display source. You must register a
 * single decoder plugin before adding audio or H.264 codecs to the VNC WiFi
 * Display Sink or invoking VNCWiFiDisplaySinkStart(). Once a decoder plugin
 * has been registered, it may not be subsequently unregistered, and no other
 * decoder plugin may be registered.
 *
 * In this API, the decoder is specified by decoderInitialize, a pointer to a
 * function which generates a factory, allowing the SDK to generate one or more
 * decoders of that type. You may alternatively register a decoder plugin with
 * VNCWiFiDisplaySinkLoadDecoderFromFile(), to have the SDK dynamically load
 * the plugin from file.
 *
 * Once registered, you may query the audio and H.264 codecs that the plugin
 * supports by calling either VNCWiFiDisplaySinkQueryDecoderAudioCodecs() or
 * VNCWiFiDisplaySinkQueryDecoderH264Codecs(). The audio and H.264 codecs that
 * you add to the VNC WiFi Display Sink with VNCWiFiDisplaySinkAddAudioCodec()
 * and VNCWiFiDisplaySinkAddH264Codec() must be supported by the decoder plugin
 * that you have registered. You needn't add every codec supported by the
 * decoder plugin, but instead choose a subset depending on your requirements.
 *
 * The VNCWiFiDisplayDecoderViewerContext passed to this method will be passed
 * unaltered to the VNCWiFiDisplayDecoderFactoryInitialize() call. This allows
 * the decoder plugin and your application to share a common context object,
 * containing anything that both need to ensure the correct display of any
 * items on the screen.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param decoderInitialize A pointer to the factory initialization function.
 * \param decoderViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the
 * VNCWiFiDisplayDecoderFactoryInitialize() function.
 *
 * \return VNCWiFiDisplayErrorNone The decoder plugin was registered
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The decoder plugin could not
 * be registered, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorAlreadyExists The decoder plugin has already been
 * registered.
 * \return VNCWiFiDisplayErrorDecoderPluginLoadFailed The decoder plugin could
 * not be registered, as it is invalid.
 *
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplaySinkQueryDecoderAudioCodecs
 * \see VNCWiFiDisplaySinkQueryDecoderH264Codecs
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkRegisterDecoder(
                 VNCWiFiDisplaySink *sink,
                 VNCWiFiDisplayDecoderFactoryInitializeType *decoderInitialize,
                 VNCWiFiDisplayDecoderViewerContext decoderViewerContext);

/**
 * \brief Registers a VNC WiFi Display Decoder Plugin for a VNC WiFi Display
 * Sink.
 *
 * This API provides an alternative method of loading a decoder plugin to that
 * used by VNCWiFiDisplaySinkRegisterDecoder(), in that it allows your
 * application to request that the SDK dynamically load the plugin from a file.
 * The decoder plugin is specified by the decoderName parameter. The SDK will
 * look in the directory specified in the
 * VNCWiFiDisplayParameterDecoderDirectory SDK parameter, and try to load a
 * decoder from a shared library with the following filename (where 'NAME' is
 * the decoderName argument):
 *
 *  * On Linux: 'libvncwifidisplaydecoder-NAME.so'
 *  * On Mac OS X: 'libvncwifidisplaydecoder-NAME.dylib'
 *  * On Windows: 'vncwifidisplaydecoder-NAME.dll'
 *
 * See \ref VNCWiFiDisplaySinkRegisterDecoder for more information on
 * registering decoder plugins.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param decoderName The name of the decoder plugin to load, as a
 * null-terminated UTF-8 string.
 * \param decoderViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the
 * VNCWiFiDisplayDecoderFactoryInitialize() function.
 *
 * \return VNCWiFiDisplayErrorNone The decoder plugin was registered
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The decoder plugin could not
 * be registered, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorNotFound The decoder plugin could not be
 * registered, as the file could not be found.
 * \return VNCWiFiDisplayErrorAlreadyExists The decoder plugin has already been
 * registered.
 * \return VNCWiFiDisplayErrorDecoderPluginLoadFailed The decoder plugin could
 * not be registered, as it is invalid.
 *
 * \see VNCWiFiDisplayParameterDecoderDirectory
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkQueryDecoderAudioCodecs
 * \see VNCWiFiDisplaySinkQueryDecoderH264Codecs
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkLoadDecoderFromFile(
                      VNCWiFiDisplaySink *sink,
                      const char *decoderName,
                      VNCWiFiDisplayDecoderViewerContext decoderViewerContext);

/**
 * \brief Sets a property of a WiFi Display decoder.
 *
 * For example, viewer applications may use this method to set the
 * visibility, position, and size of a WiFi Display decoder's overlay.
 *
 * The arguments to this method will be passed directly to the decoder's
 * VNCWiFiDisplayDecoderPropertySet() method.
 *
 * If there is no WiFi Display decoder loaded, or the WiFi Display decoder
 * failed to initialise, calls to this method will be ignored. This method has
 * no side effects other than passing the key and value directly to the loaded
 * WiFi Display decoder. Therefore, it is advisable to wait for
 * VNCWiFiDisplaySinkDecoderCreatedCallback() before making use of this API.
 *
 * \param pViewer The VNC Viewer which is requesting the change.
 * \param propertyKey The key representing the property being set.
 * \param newValue A pointer to the new value of the property. This is owned
 * by the viewer application, and should be valid for the duration of
 * this method call.
 * \param valueSize The size of the value pointed to by newValue, in bytes.
 * \return VNCWiFiDisplayErrorNone if the change was successful, or an
 * appropriate error message if it was not successful.
 *
 * \see VNCWiFiDisplayDecoderPropertySet
 * \see VNCWiFiDisplaySinkDecoderCreatedCallback
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetDecoderProperty(VNCWiFiDisplaySink *sink,
                                     VNCWiFiDisplayDecoderProperty propertyKey,
                                     const void* newValue,
                                     size_t valueSize);
/**
 * \brief Queries the registered VNC WiFi Display Decoder Plugin for the set of
 * audio codecs that it supports.
 *
 * If the VNC WiFi Display Decoder Plugin does not support decoding of audio
 * streams, then it shall report no audio codecs. Otherwise, the plugin shall
 * always provide an audio codec which supports the audio format
 * VNCWiFiDisplayAudioFormatLPCM and includes
 * VNCWiFiDisplayAudioModeLPCM_48000_16_2 as one of its supported audio modes.
 * The plugin may also report any additional audio codecs, if it supports them.
 *
 * Ownership of the memory pointed to by 'codecs' remains with the SDK.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param codecs On success, set to point to an array of
 * VNCWiFiDisplayAudioCodec instances describing the audio codecs that the
 * decoder supports, or NULL if no audio codecs are supported. If an array is
 * provided, then ownership of this transfers to your application.
 * \param codecSize On success, set to the size of an individual
 * VNCWiFiDisplayAudioCodec instance.
 * \param codecCount On success, set to the number of audio codecs supported.
 *
 * \return VNCWiFiDisplayErrorNone The decoder plugin was queried successfully.
 * \return VNCWiFiDisplayErrorMissingDecoderPlugin No decoder plugin has yet
 * been registered.
 *
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplayAudioCodec, VNCWiFiDisplaySinkAddAudioCodec
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkQueryDecoderAudioCodecs(VNCWiFiDisplaySink *sink,
                                          const VNCWiFiDisplayAudioCodec **codecs,
                                          size_t *codecSize,
                                          size_t *codecCount);

/**
 * \brief Queries the registered VNC WiFi Display Decoder Plugin for the set of
 * H.264 codecs that it supports.
 *
 * The plugin shall always provide an H.264 codec which supports the CBP
 * profile, and a resolution of 640x480 p60 at H.264 level 3.1. If the plugin
 * supports a higher resolution in the 60Hz family, then it shall also provide
 * a H.264 codec which supports the CBP profile, and a resolution of 720x480
 * p60 at H.264 level 3.1.  If the plugin supports a higher resolution in the
 * 50Hz family, then it shall also provide a H.264 codec which supports the CBP
 * profile, and a resolution of 720x576 p50 at H.264 level 3.1. The plugin may
 * also report any additional audio codecs, if it supports them.
 *
 * Ownership of the memory pointed to by 'codecs' remains with the SDK.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param codecs On success, set to point to an array of
 * VNCWiFiDisplayH264Codec instances describing the H.264 codecs that the
 * decoder supports, or NULL if no H.264 codecs are supported. If an array is
 * provided, then ownership of this transfers to your application.
 * \param codecSize On success, set to the size of an individual
 * VNCWiFiDisplayH264Codec instance.
 * \param codecCount On success, set to the number of H.264 codecs supported.
 *
 * \return VNCWiFiDisplayErrorNone The decoder plugin was queried successfully.
 * \return VNCWiFiDisplayErrorMissingDecoderPlugin No decoder plugin has yet
 * been registered.
 *
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplayH264Codec, VNCWiFiDisplaySinkAddH264Codec
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkQueryDecoderH264Codecs(VNCWiFiDisplaySink *sink,
                                         const VNCWiFiDisplayH264Codec **codecs,
                                         size_t *codecSize,
                                         size_t *codecCount);

/**
 * \brief Adds an audio codec to a VNC WiFi Display Sink.
 *
 * You should call this method if you wish to support the decoding of audio
 * streams from the WiFi Display source. In this SDK, an audio codec is
 * represented by the audio format that it supports. An audio codec must also
 * declare at least one supported audio mode. The audio codec that you add must
 * be supported by the VNC WiFi Display Decoder Plugin that you registered with
 * VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile().
 *
 * WiFi Display sources and sinks that are audio-capable must include an audio
 * codec which supports the audio format VNCWiFiDisplayAudioFormatLPCM and
 * includes VNCWiFiDisplayAudioModeLPCM_48000_16_2 as one of its supported
 * audio modes. No two audio codecs may support the same audio format.
 *
 * Audio codecs may not be added whilst the VNC WiFi Display Sink is running.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param codec The audio codec to add.
 * \param codecSize The size of codec.
 *
 * \return VNCWiFiDisplayErrorNone The audio codec was added successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The audio codec could not be
 * added, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The audio codec could not be
 * added, as it has an invalid audio format or audio modes.
 * \return VNCWiFiDisplayErrorAlreadyExists An audio codec with the format
 * given has already been added.
 * \return VNCWiFiDisplayErrorMissingAudioMode The audio codec could not be
 * added, as it supports the VNCWiFiDisplayAudioFormatLPCM format but not the
 * mandatory VNCWiFiDisplayAudioModeLPCM_48000_16_2 mode.
 * \return VNCWiFiDisplayErrorMissingDecoderPlugin The audio codec could not be
 * added, as either no decoder plugin has been registered, or the registered
 * decoder plugin does not support it.
 *
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplaySinkQueryDecoderAudioCodecs
 * \see VNCWiFiDisplayAudioCodec, VNCWiFiDisplayAudioFormat
 * \see VNCWiFiDisplaySinkClearAudioCodecs
 * \see AudioModeLPCM
 * \see AudioModeAAC
 * \see AudioModeAC3
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkAddAudioCodec(VNCWiFiDisplaySink *sink,
                                const VNCWiFiDisplayAudioCodec *codec,
                                size_t codecSize);

/**
 * \brief Clears all audio codecs currently associated with a VNC WiFi Display
 * Sink.
 *
 * This clears all audio codecs that were added with
 * VNCWiFiDisplaySinkAddAudioCodec(). If no audio codecs have been added, then
 * this method does nothing. Audio codecs may not be removed whilst the VNC
 * WiFi Display Sink is running.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \return VNCWiFiDisplayErrorNone All audio codecs were cleared successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning Audio codecs could not be
 * cleared, as the WiFi Display sink thread is running.
 *
 * \see VNCWiFiDisplaySinkAddAudioCodec
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkClearAudioCodecs(VNCWiFiDisplaySink *sink);

/**
 * \brief Adds a H.264 video codec to a VNC WiFi Display Sink.
 *
 * You must specify at least one H.264 codec before invoking
 * VNCWiFiDisplaySinkStart(). In this SDK, a H.264 codec is represented by its
 * properties. The H.264 codec that you add must be supported by the VNC WiFi
 * Display Decoder Plugin that you registered with
 * VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile().
 *
 * Primary WiFi Display sinks must include a H.264 codec which supports the CBP
 * profile, and a resolution of 640x480 p60 at H.264 level 3.1.
 *
 * If a higher resolution is supported in the 60Hz family, then a sink must
 * also include a H.264 codec which supports the CBP profile, and a resolution
 * of 720x480 p60 at H.264 level 3.1.
 *
 * If a higher resolution is supported in the 50Hz family, then the sink must
 * also include a H.264 codec which supports the CBP profile, and a resolution
 * of 720x576 p50 at H.264 level 3.1.
 *
 * H.264 codecs may not be added whilst the VNC WiFi Display Sink is running.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param codec The H.264 codec to add.
 * \param codecSize The size of codec.
 *
 * \return VNCWiFiDisplayErrorNone The H.264 codec was added successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The H.264 codec could not be
 * added, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The H.264 codec could not be
 * added, as one or more details are invalid.
 * \return VNCWiFiDisplayErrorAlreadyExists A H.264 codec has already been
 * added with the same details.
 * \return VNCWiFiDisplayErrorMissingDecoderPlugin The audio codec could not be
 * added, as either no decoder plugin has been registered, or the registered
 * decoder plugin does not support it.
 *
 * \see VNCWiFiDisplaySinkRegisterDecoder
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 * \see VNCWiFiDisplaySinkQueryDecoderH264Codecs
 * \see VNCWiFiDisplayH264Profile, VNCWiFiDisplayH264Level
 * \see VNCWiFiDisplayH264Codec, VNCWiFiDisplaySinkClearH264Codecs
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkAddH264Codec(VNCWiFiDisplaySink *sink,
                               const VNCWiFiDisplayH264Codec *codec,
                               size_t codecSize);

/**
 * \brief Clears all H.264 codecs currently associated with a VNC WiFi Display
 * Sink.
 *
 * This clears all H.264 codecs that were added with
 * VNCWiFiDisplaySinkAddH264Codec(). If no H.264 codecs have been added, then
 * this method does nothing. H.264 codecs may not be removed whilst the VNC
 * WiFi Display Sink is running.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \return VNCWiFiDisplayErrorNone All H.264 codecs were cleared successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning H.264 codecs could not be
 * cleared, as the WiFi Display sink thread is running.
 *
 * \see VNCWiFiDisplaySinkAddH264Codec
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkClearH264Codecs(VNCWiFiDisplaySink *sink);

/**
 * \brief Sets the MirrorLink configuration that a VNC WiFi Display Sink should
 * use for User Input Back Channel (UIBC) sessions.
 *
 * A VNC WiFi Display Sink can optionally support a UIBC connection with a
 * MirrorLink WiFi Display source. To enable this, call this method to specify
 * the MirrorLink configuration that you support. Note that if you are
 * implementing a MirrorLink client, then UIBC support is mandatory.
 *
 * The MirrorLink configuration for UIBC sessions may not be set whilst the VNC
 * WiFi Display Sink is running.
 *
 * The MirrorLink configuration that this sink supports (if any) is reported to
 * the WiFi Display source through the wfd-uibc-capability-parameter RTSP. If
 * the WiFi source supports UIBC MirrorLink input, it shall select a
 * configuration that it supports, and which is compatible with that presented
 * by the sink. The SDK shall notify your application through the
 * VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback().
 *
 * The WiFi Display source may enable or disable UIBC connection by
 * setting the wfd-uibc-setting RTSP parameter. The SDK shall notify your
 * application when the UIBC connection state changes through the
 * VNCWiFiDisplaySinkUIBCStateChangedCallback().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param configuration The MirrorLink configuration to set.
 * \param configurationSize The size of configuration.
 *
 * \return VNCWiFiDisplayErrorNone The UIBC MirrorLink configuration was set
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The UIBC MirrorLink
 * configuration could not be set, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The UIBC MirrorLink
 * configuration could not be set, as the configuration details are invalid.
 *
 * \see VNCWiFiDisplayUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration(
                VNCWiFiDisplaySink *sink,
                const VNCWiFiDisplayUIBCMirrorLinkConfiguration *configuration,
                size_t configurationSize);

/**
 * \brief Sets the WiFi Display source that a VNC WiFi Display Sink should
 * connect to.
 *
 * You must call this method to specify the IP address and control port of the
 * WiFi Display source. The IP address must be specified using the string
 * format "a.b.c.d", where each component is a decimal value representing a
 * single byte of the IP address and the bytes are assigned in left-to-right
 * order to produce the corresponding binary address. For example,
 * "192.168.0.1" is a valid string formatted IP address.
 *
 * You must call this method to specify this information before invoking
 * VNCWiFiDisplaySinkStart(). This method may not be called while the WiFi
 * Display sink thread is running.
 *
 * Since discovery of WiFi Display sources is not supported by this SDK, you
 * must obtain the details of the WiFi Display source separately. 
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param sourceAddress The IP address of the WiFi Display source, as a
 * null-terminated UTF-8 string.
 * \param controlPort The TCP port that the WiFi Display source is listening on
 * for incoming RTSP connections.
 *
 * \return VNCWiFiDisplayErrorNone The WiFi Display source details were
 * accepted.
 * \return VNCWiFiDisplayErrorIllegalWhileRunning The WiFi Display source
 * details could not be accepted, as the WiFi Display sink thread is running.
 * \return VNCWiFiDisplayErrorInvalidParameter The WiFi Display source details
 * could not be accepted, as they are invalid.
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSetSource(VNCWiFiDisplaySink *sink,
                            const char *sourceAddress,
                            vnc_uint16_t controlPort);

/**
 * \brief Starts a VNC WiFi Display Sink.
 *
 * This starts the WiFi Display sink thread. The VNC WiFi Display Sink shall
 * open an RTSP connection with the WiFi Display source and establish a WiFi
 * Display session. You must first add at least one license with
 * VNCWiFiDisplaySinkAddLicense(), and set the native resolution and active
 * connector type with VNCWiFiDisplaySinkSetNativeResolution() and
 * VNCWiFiDisplaySinkSetActiveConnectorType(). It is also necessary to register
 * a VNC WiFi Display Decoder Plugin with VNCWiFiDisplaySinkRegisterDecoder()
 * or VNCWiFiDisplaySinkLoadDecoderFromFile(), and add at least one H.264
 * codec, supported by the registered decoder plugin, with
 * VNCWiFiDisplaySinkAddH264Codec(). Finally, you must specify the WiFi Display
 * source that you wish to connect to with VNCWiFiDisplaySinkSetSource(). Once
 * you have performed these tasks, this method may be invoked.
 *
 * When started, the WiFi Display sink thread will attempt to establish a WiFi
 * Display session with the WiFi Display source. If this succeeds, the SDK
 * shall call VNCWiFiDisplaySinkStateChangedCallback() with
 * VNCWiFiDisplayStatePlaying to notify your application. If you are
 * implementing a MirrorLink client, then you should immediately request that
 * the RTP stream from the WiFi Display source be paused by calling
 * VNCWiFiDisplaySinkPause(). Call VNCWiFiDisplaySinkResume() to resume the RTP
 * stream, once the first application has been launched.
 *
 * Either the WiFi Display source, or your application, may request that the
 * RTP stream be paused at any time. When this occurs, the SDK shall call
 * VNCWiFiDisplaySinkStateChangedCallback() with VNCWiFiDisplayStatePaused. Your
 * application may call VNCWiFiDisplaySinkResume() to resume the stream, or
 * alternatively the WiFi Display source may request that the stream be
 * resumed. When this occurs, the SDK call shall
 * VNCWiFiDisplaySinkStateChangedCallback() with VNCWiFiDisplayStatePlaying, to
 * notify your application that the RTP stream has resumed.
 *
 * When the WiFi Display sink thread is about to stop,
 * VNCWiFiDisplaySinkErrorCallback() shall be invoked with an error code
 * describing the reason for the stoppage. The WiFi Display sink thread will
 * stop when a non-recoverable error occurs, the WiFi Display source tears down
 * the WiFi Display session, or either VNCWiFiDisplaySinkStop() or
 * VNCWiFiDisplaySinkDestroy() is called.
 *
 * If the WiFi Display sink thread is already running, this method does
 * nothing. 
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \return VNCWiFiDisplayErrorNone The WiFi Display Sink thread is now running.
 * \return VNCWiFiDisplayErrorNoNativeResolution The WiFi Display Sink thread
 * could not be started, as VNCWiFiDisplaySinkSetNativeResolution() has not
 * been called.
 * \return VNCWiFiDisplayErrorNoActiveConnectorType The WiFi Display Sink
 * thread could not be started, as VNCWiFiDisplaySinkSetActiveConnectorType()
 * has not been called.
 * \return VNCWiFiDisplayErrorMissingAudioCodec The WiFi Display Sink thread
 * could not be started, as a mandatory audio codec has not been added.
 * \return VNCWiFiDisplayErrorMissingH264Codec The WiFi Display Sink thread
 * could not be started, as a mandatory H.264 codec has not been added.
 * \return VNCWiFiDisplayErrorNoSource The WiFi Display Sink thread could not
 * be started, as no WiFi Display source has been specified.
 *
 * \see VNCWiFiDisplaySinkAddLicense, VNCWiFiDisplaySinkSetNativeResolution
 * \see VNCWiFiDisplaySinkSetActiveConnectorType
 * \see VNCWiFiDisplaySinkAddAudioCodec, VNCWiFiDisplaySinkAddH264Codec
 * \see VNCWiFiDisplaySinkSetSource, VNCWiFiDisplaySinkStateChangedCallback
 * \see VNCWiFiDisplaySinkPause, VNCWiFiDisplaySinkStop
 * \see VNCWiFiDisplaySinkDestroy
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkStart(VNCWiFiDisplaySink *sink);

/**
 * \brief Stops a VNC WiFi Display Sink.
 *
 * This stops the WiFi Display sink thread asynchronously.
 * VNCWiFiDisplaySinkErrorCallback() will be invoked with the error
 * VNCWiFiDisplayErrorStopped when the WiFi Display sink thread is about to
 * stop.
 *
 * Calling VNCWiFiDisplaySinkStop() does not destroy the VNC WiFi Display Sink
 * object itself. You may either call VNCWiFiDisplaySinkDestroy() to destroy
 * it, or, once it has finished stopping, call VNCWiFiDisplaySinkStart() to
 * start it again. You may wish to set a different WiFi Display source by 
 * calling VNCWiFiDisplaySinkSetSource() again first, however.
 *
 * If the Wifi Display sink thread is not running, this method does nothing.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \see VNCWiFiDisplaySinkStart, VNCWiFiDisplaySinkDestroy
 * \see VNCWiFiDisplaySinkErrorCallback
 */
typedef void VNCCALL
VNCWiFiDisplaySinkStop(VNCWiFiDisplaySink *sink);

/**
 * \brief Pauses the RTP media stream received by a VNC WiFi Display Sink.
 *
 * This sends an RTSP PAUSE request to the WiFi Display source, to request that
 * the RTP media stream be paused. Once the RTP media stream has been paused,
 * the SDK shall invoke VNCWiFiDisplaySinkStateChangedCallback() with
 * VNCWiFiDisplayStatePaused to notify your application that the media stream is
 * no longer playing.
 *
 * To resume the RTP media stream, call VNCWiFiDisplaySinkResume().
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \return VNCWiFiDisplayErrorNone The RTP media stream was paused
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The RTP media stream could
 * not be paused, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidState The RTP media stream cannot be
 * paused, because the sink is not in the VNCWiFiDisplayStatePlaying state.
 *
 * \see VNCWiFiDisplaySinkStateChangedCallback, VNCWiFiDisplaySinkResume
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkPause(VNCWiFiDisplaySink *sink);

/**
 * \brief Resumes a paused VNC WiFi Display Sink.
 *
 * This sends an RTSP PLAY request to the WiFi Display source, to request that
 * the paused RTP media stream be resumed. Once the RTP media stream has been
 * resumed, the SDK shall invoke VNCWiFiDisplaySinkStateChangedCallback() with
 * VNCWiFiDisplayStatePlaying to notify your application that the media stream
 * is now playing again.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \return VNCWiFiDisplayErrorNone The RTP media stream was resumed
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The RTP media stream could
 * not be resumed, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidState The RTP media stream cannot be
 * paused, because the sink is not in the VNCWiFiDisplayStatePaused state.
 *
 * \see VNCWiFiDisplaySinkStateChangedCallback
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkResume(VNCWiFiDisplaySink *sink);

/**
 * \brief Informs the SDK that a VNC WiFi Display Sink has lost the input
 * focus.
 *
 * Calling this method cancels any touch, pointer or key events currently in
 * progress. If the application has used any of these, then it must call this
 * method whenever the viewer UI loses the input focus.
 *
 * \param sink The VNC WiFi Display Sink instance.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkKeyEvent
 * \see VNCWiFiDisplaySinkSendMirrorLinkDeviceKeyEvent
 * \see VNCWiFiDisplaySinkSendMirrorLinkPointerEvent
 * \see VNCWiFiDisplaySinkSendMirrorLinkTouchEvent
 */
typedef void VNCCALL
VNCWiFiDisplaySinkLostFocus(VNCWiFiDisplaySink *sink);

/**
 * \brief Sends a MirrorLink key event over a VNC WiFi Display Sink's UIBC
 * MirrorLink connection.
 *
 * This API implements the same behaviour as VNCViewerSendDeviceKeyOrKeyEvent()
 * in the VNC Viewer SDK for MirrorLink connections. Key event mappings are
 * performed by the MirrorLink WiFi Display source.
 *
 * Your application should also ensure that it calls
 * VNCWiFiDisplaySinkLostFocus() whenever your application loses focus. This
 * will automatically communicate to the MirrorLink WiFi Display source that an
 * ongoing series of key events has been cancelled.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param keyDown Non-zero if a key-down event should be sent, zero if a key-up
 * event should be sent.
 * \param keySymbol The X key symbol (e.g. XK_Tab) representing the key that has
 * been pressed or released. These constants are defined in keysymdef.h.
 *
 * \return VNCWiFiDisplayErrorNone The key event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The key event could not be
 * sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The key event could not be sent,
 * because the sink's UIBC state is not VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkLostFocus
 * \see keysymdef.h
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkKeyEvent(VNCWiFiDisplaySink *sink,
                                         vnc_uint8_t keyDown,
                                         vnc_int32_t keySymbol);

/**
 * \brief Sends a MirrorLink device key event over a VNC WiFi Display Sink's
 * UIBC connection.
 *
 * This API implements the same behaviour as VNCViewerSendDeviceKey()
 * in the VNC Viewer SDK for MirrorLink connections. Abstracted button codes
 * are mapped to the appropriate MirrorLink key symbol and sent to the MirrorLink
 * WiFi Display source as a regular Key Event.
 *
 * Your application should also ensure that it calls
 * VNCWiFiDisplaySinkLostFocus() whenever your application loses focus. This
 * will automatically communicate to the MirrorLink WiFi Display source that an
 * ongoing series of key events has been cancelled.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param keyCode The abstracted button code.
 * \param pressed Non-zero if a key-down event should be sent, zero if a key-up
 * event should be sent.
 *
 * \return VNCWiFiDisplayErrorNone The device key event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The device key event could
 * not be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The device key event could not be
 * sent, because the sink's UIBC state is not VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkLostFocus
 * \see keysymdef.h
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkDeviceKeyEvent(VNCWiFiDisplaySink *sink,
                                               VNCDeviceKey keyCode,
                                               vnc_uint8_t pressed);

/**
 * \brief Sends a MirrorLink pointer event over a VNC WiFi Display Sink's UIBC
 * MirrorLink connection.
 *
 * This API implements the same behaviour as VNCViewerSendPointerEvent() in the
 * VNC Viewer SDK for MirrorLink connections. If the UIBC connection is active,
 * then you should call this method whenever the user moves the local mouse
 * pointer or changes its button state, provided that the pointer is over the
 * area of the screen in which the remote display is being rendered. The
 * co-ordinates that you provide must be mapped appropriately to the remote display.
 *
 * A call to this method informs the MirrorLink WiFi Display source of the
 * pointer's current state. The source maintains the pointer in this state
 * until either the next call to this method or the end of the session.
 * Therefore, if you want to cause the source to click and then release a mouse
 * button, you must make two calls to this method.
 *
 * Your application should also ensure that it calls
 * VNCWiFiDisplaySinkLostFocus() whenever your application loses focus. This
 * will automatically communicate to the MirrorLink WiFi Display source that an
 * ongoing series of pointer events has been cancelled.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param buttonMask A bit-mask representing the pointer device buttons that
 * are currently depressed. Use the VNCPointerDeviceButton enumeration to set
 * the bits in buttonMask.
 * \param x The x co-ordinate of the pointer.
 * \param y The y co-ordinate of the pointer.
 *
 * \return VNCWiFiDisplayErrorNone The pointer event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The pointer event could
 * not be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The pointer event could not be
 * sent, because the sink's UIBC state is not VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkLostFocus
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkPointerEvent(VNCWiFiDisplaySink *sink,
                                             VNCPointerDeviceButton buttonMask,
                                             vnc_uint16_t x,
                                             vnc_uint16_t y);

/**
 * \brief Sends a MirrorLink touch event over a VNC WiFi Display Sink's UIBC
 * MirrorLink connection.
 *
 * This API implements the same behaviour as VNCViewerSendTouchEvent() in the
 * VNC Viewer SDK for MirrorLink connections. Touch events differ from pointer
 * events in that they encapsulate multiple locations (usually of fingertips)
 * with different pressures, whereas a pointer event has a single point of
 * contact (usually of a mouse pointer) with a combination of different
 * buttons.
 *
 * A touch event is described by an array of VNCTouchDescriptor structures.
 * Each structure gives the location of a single point of contact, together
 * with the pressure applied and an integer identifier that allows the server
 * to track that particular point of contact as it moves.
 * 
 * Your application must ensure that, when a point of contact is lost,
 * this method is called again with a pressure of zero for that point of
 * contact.
 * 
 * Every time you call this API you must include all points of contact that
 * belong to the current gesture, even if only some of them have changed state
 * since the last call.
 *
 * Your application should also ensure that it calls
 * VNCWiFiDisplaySinkLostFocus() whenever the viewer application loses focus.
 * This will automatically communicate to the MirrorLink WiFi Display source
 * that an ongoing series of touch events has been cancelled (some platforms
 * regard 'cancelled' and 'finished' as separate outcomes for gesture
 * recognition).
 *
 * If you want to use the timestamps provided by the touch-screen then you can
 * pass them to the SDK through the timestampSeconds and timestampNanoseconds
 * parameters. These values should represent the absolute time at which the
 * events were recorded. If instead you want the SDK to handle the timestamps
 * then simply set the two parameters to 0.
 * 
 * A maximum of 255 concurrent touches are supported by this API. The
 * touchDescriptorCount should always be less than or equal to this limit.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param touchDescriptors The array of VNCTouchDescriptors that specify the
 * touch event.
 * \param touchDescriptorCount The number of VNCTouchDescriptors in the
 * touchDescriptors array.
 * \param timestampSeconds The timestamp of the event in seconds.
 * \param timestampNanoseconds The timestamp of the event in nanoseconds.
 *
 * \return VNCWiFiDisplayErrorNone The touch event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The touch event could not
 * be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The touch event could not be sent,
 * because the sink's UIBC state is not VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkLostFocus
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkTouchEvent(
                                    VNCWiFiDisplaySink *sink,
                                    const VNCTouchDescriptor *touchDescriptors,
                                    size_t touchDescriptorCount,
                                    vnc_uint32_t timestampSeconds,
                                    vnc_uint32_t timestampNanoseconds);

/**
 * \brief Sends a MirrorLink sink status event over a VNC WiFi Display Sink's
 * UIBC connection.
 *
 * This allows a MirrorLink WiFi Display sink to communicate its current device
 * status to a MirrorLink WiFi Display source. If the source sends a status
 * event, then the SDK shall call
 * VNCWiFiDisplaySinkMirrorLinkSourceStatusEventCallback() to notify your
 * application.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param deviceStatus A bit-mask representing the current device status. Use
 * the VNCDeviceStatusFeature enumeration to set the bits in buttonMask.
 * \param deviceStatusSize The size of deviceStatus.
 *
 * \return VNCWiFiDisplayErrorNone The sink status event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The sink status event
 * could not be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The sink status event could not
 * be sent, because the sink's UIBC state is not
 * VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkSinkStatusEvent(
                                           VNCWiFiDisplaySink *sink,
                                           const VNCDeviceStatus *deviceStatus,
                                           size_t deviceStatusSize);

/**
 * \brief Sends a MirrorLink UI blocking event over a VNC WiFi Display Sink's
 * UIBC connection.
 *
 * This allows a MirrorLink WiFi Display sink to request that the incoming
 * video stream, or some portion of it, be blocked. A UI blocking event may be
 * associated with a single UI context event received from the MirrorLink WiFi
 * Display source, if desired.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param contextReference The reference of the received UI context event that
 * this UI blocking event is associated with, or zero if the blocking event is
 * not associated with a UI context event.
 * \param info An array of UI blocking information entries for this event.
 * \param infoSize The size of a single entry in info.
 * \param infoCount The number of entries in info.
 *
 * \return VNCWiFiDisplayErrorNone The UI blocking event was sent successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The UI blocking event
 * could not be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The UI blocking event could not
 * be sent, because the sink's UIBC state is not
 * VNCWiFiDisplaySinkUIBCStateActive.
 * \return VNCWiFiDisplayErrorUnknownMirrorLinkContextEvent The UI blocking
 * event could not be sent, because contextReference is non-zero and does not
 * refer to a known UI context event.
 * \return VNCWiFiDisplayErrorInvalidParameter The UI blocking event could not
 * be sent, because one or more UI blocking information entries are invalid.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplayMirrorLinkUIBlockingInformation
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent(
                     VNCWiFiDisplaySink *sink,
                     vnc_uint32_t contextReference,
                     const VNCWiFiDisplayMirrorLinkUIBlockingInformation *info,
                     size_t infoSize,
                     size_t infoCount);

/**
 * \brief Sends a MirrorLink audio blocking event over a VNC WiFi Display
 * Sink's UIBC connection.
 *
 * This allows a MirrorLink WiFi Display sink to request that the incoming
 * audio stream, or some portion of it, be blocked. Each audio blocking event
 * may be associated with a single audio context event received from the
 * MirrorLink WiFi Display source, if desired.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param contextReference The reference of the received audio context event
 * that this audio blocking event is associated with, or zero if the blocking
 * event is not associated with an audio context event.
 * \param info An array of audio blocking information entries for this event.
 * \param infoSize The size of a single entry in info.
 * \param infoCount The number of entries in info.
 *
 * \return VNCWiFiDisplayErrorNone The audio blocking event was sent
 * successfully.
 * \return VNCWiFiDisplayErrorIllegalWhileNotRunning The audio blocking event
 * could not be sent, as the WiFi Display sink thread is not running.
 * \return VNCWiFiDisplayErrorInvalidUIBCState The audio blocking event could
 * not be sent, because the sink's UIBC state is not
 * VNCWiFiDisplaySinkUIBCStateActive.
 * \return VNCWiFiDisplayErrorUnknownMirrorLinkContextEvent The audio blocking
 * event could not be sent, because contextReference is non-zero and does not
 * refer to a known audio context event.
 * \return VNCWiFiDisplayErrorInvalidParameter The audio blocking event could
 * not be sent, because one or more audio blocking information entries are
 * invalid.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplayMirrorLinkAudioBlockingInformation
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent(
                  VNCWiFiDisplaySink *sink,
                  vnc_uint32_t contextReference,
                  const VNCWiFiDisplayMirrorLinkAudioBlockingInformation *info,
                  size_t infoSize,
                  size_t infoCount);

/**
 * \brief Sends a MirrorLink sink cut text event over a VNC WiFi Display Sink's
 * UIBC connection.
 *
 * This API implements the same behaviour as VNCViewerSendClientCutText() in
 * the VNC Viewer SDK for MirrorLink connections. If your application is
 * running on a platform that supports a clipboard, and you are able to observe
 * the clipboard's contents, then you may wish to call this method when the
 * clipboard has new text copied to it. This will cause the VNC server host's
 * clipboard contents to be updated to match the new text, allowing the user to
 * copy text from a local application to the framebuffer.
 *
 * The text sent to the source can be truncated before sending, depending on
 * the value of the VNCWiFiDisplayParameterMaxCutText parameter.
 *
 * This method requires an active UIBC connection, and so may only be called
 * when the current UIBC state is VNCWiFiDisplaySinkUIBCStateActive.
 * 
 * The WiFi Display Source shall only handle cut text events if it has indicated
 * support for them when selecting the UIBC MirrorLink configuration.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param clipboardText The text to be copied to the MirrorLink WiFi Display
 * source host's clipboard. This should be UTF-8 encoded, but need not be
 * null-terminated.
 * \param clipboardTextLength The length of clipboardText in bytes.
 *
 * \see VNCWiFiDisplaySinkMirrorLinkSourceCutTextEventCallback
 * \see VNCWiFiDisplayParameterMaxCutText
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent(VNCWiFiDisplaySink *sink,
                                                 const char *clipboardText,
                                                 size_t clipboardTextLength);

/**
 * \brief Translates a Unicode character into an X key symbol.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param unicodeCharacter A Unicode character in native-endian UTF-32 encoding.
 *
 * \return The X key symbol corresponding to unicodeCharacter, or
 * (vnc_uint32_t) -1 if there is none.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCWiFiDisplaySinkUnicodeToXKeySymbol(VNCWiFiDisplaySink *sink, vnc_uint32_t unicodeCharacter);

/**
 * \brief Translates an X key symbol into a Unicode character.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param keySymbol An X key symbol.
 *
 * \return A Unicode character, in native-endian UTF-32 encoding, corresponding to
 * keySymbol, or (vnc_uint32_t) -1 if there is none.
 *
 * \see VNCWiFiDisplaySinkSendMirrorLinkKeyEvent, keysymdef.h
 */
typedef vnc_uint32_t VNCCALL
VNCWiFiDisplaySinkXKeySymbolToUnicode(VNCWiFiDisplaySink *sink, vnc_uint32_t keySymbol);

#ifdef __cplusplus
}
#endif

#endif  // UUID_a7efdd62_01c8_11e4_9403_f36b20d3dfa8
